% This script reads imaris files and measures the mRNA distribution
% relative to each centrosome. Written by Ingrid Brust-Mascher and Xueer
% Jiang, Jao lab, University of California, Davis (April 2019). Publications
% that have utilized this script should acknowledge the resource and cite
% our Bio-Protocol Paper.






clear
% define fpath: location of a folder containing the processed imaris images
fpath='/Users/jao_lab/Documents/MATLAB/bio_protocol_images_control';       

% define protein surface of interest and RNA spot of interest for
% quantification
protein_surface_number = 2;
spot_number = 1;

% define the decon channel of RNA spot of interest
decon_RNA_channel_number = '1';

% define intensity sum channel number for generalization
intensity_sum = 'Intensity Sum Channel';
decon_channel_RNA_intensity_sum = [intensity_sum, ' ', decon_RNA_channel_number];

% use dir function to list folder contents and select imaris files
ImarisImages=dir(fullfile(fpath,'*.ims'));

% define numbers of files
NumberImages=length(ImarisImages); 

% Create a forloop to define filenames for analysis
for k=1:NumberImages                   
   
   % filename to analyze
   filename=ImarisImages(k).name;            
   
   % create filenames
   xPath=fullfile(fpath,filename); 
    
   % read imaris files using ImarisReader function
   imsObj=ImarisReader(xPath);             
   
   % index surface statisitics of protein of interest
   SurfaceStatistics=imsObj.Surfaces(protein_surface_number).GetStatistics; 
    
   % index spot statisitics of RNA of interest
   SpotStatistics=imsObj.Spots(spot_number).GetStatistics;  

   % from SurfaceStatistics get centrosome coordinates centro (2x3 matrix)
  
   % define number of surface statisitics
   NumberSurStats=length(SurfaceStatistics);   
   
   for i=1:NumberSurStats
       % extract volume and position statistics
       if strcmp (SurfaceStatistics(i).Name,"Volume") 
           Volume=SurfaceStatistics(i).Value;
        elseif strcmp (SurfaceStatistics(i).Name,"Position X") 
           PositionX=SurfaceStatistics(i).Value;
        elseif strcmp (SurfaceStatistics(i).Name,"Position Y") 
           PositionY=SurfaceStatistics(i).Value;
        elseif strcmp (SurfaceStatistics(i).Name,"Position Z") 
           PositionZ=SurfaceStatistics(i).Value;
       end
   end
   
   % define centrosome as the 2 largest centrosomal protein surface volume
   % and extract positions of the 2 centrosome.
   NumberProteinSurfaces=size(Volume);
   centro=zeros(2,3);
   largest=0;
   for j=1:2
       for i=1:NumberProteinSurfaces
           if Volume(i)==max(Volume)
               centro(j,1)=PositionX(i);
               centro(j,2)=PositionY(i);
               centro(j,3)=PositionZ(i);
               largest=i;
           end
       end
       Volume(largest)=0;
   end
        
   % from SpotStatistics get a new file with posX, posY, posZ, intensity
   % sum of RNA channel
   
   % define number of spot statisitics
   NumberSpotStats=length(SpotStatistics); 

   for i=1:NumberSpotStats
       if strcmp (SpotStatistics(i).Name,"Position X") 
           PositionX=SpotStatistics(i).Value;
        elseif strcmp (SpotStatistics(i).Name,"Position Y") 
           PositionY=SpotStatistics(i).Value;
        elseif strcmp (SpotStatistics(i).Name,"Position Z") 
           PositionZ=SpotStatistics(i).Value;
        elseif strcmp (SpotStatistics(i).Name,decon_channel_RNA_intensity_sum)
           IntensitySum=SpotStatistics(i).Value;
       end
   end
   RNAfile=[PositionX,PositionY,PositionZ,IntensitySum];

   NumberRNASpots=size(RNAfile,1); 
   result=zeros(NumberRNASpots,4);
   
   %generate distances and choose closest centrosome
   for i=1:NumberRNASpots
       result(i,1)=RNAfile(i,4);                           %this should be spot intensity selected above
       dist1=sqrt((RNAfile(i,1)-centro(1,1))^2+(RNAfile(i,2)-centro(1,2))^2+(RNAfile(i,3)-centro(1,3))^2);
       dist2=sqrt((RNAfile(i,1)-centro(2,1))^2+(RNAfile(i,2)-centro(2,2))^2+(RNAfile(i,3)-centro(2,3))^2);
       if dist1<dist2
           result(i,2)=1;
           result(i,3)=dist1;
       else
           result(i,2)=2;
           result(i,3)=dist2;
       end
   end
    
   %generate his1gram with 40 bins of 0.5 micron size
   his1=zeros(40,2);

   for j=1:40
      for i=1:NumberRNASpots
          if result(i,2)==1
              if result(i,3)>=0.5*(j-1)
                  if result(i,3)<0.5*j
                     his1(j,1)=his1(j,1)+result(i,1);
                  end
              end
          else
              if result(i,3)>=0.5*(j-1)
                  if result(i,3)<0.5*j
                      his1(j,2)=his1(j,2)+result(i,1);
                  end
              end
          end
      end
   end
   
   
% write out histogram file as CSV file
table_result = array2table(his1);
table_result.Properties.VariableNames{'his11'} = 'first_centrosome';
table_result.Properties.VariableNames{'his12'} = 'second_centrosome';
filetype = '.CSV';
OutPutName = [filename filetype]; 
OutputPath = fullfile(fpath, OutPutName);
writetable(table_result,OutputPath);

end